/**************************************************************************
 *
 * Copyright 2010, 2011 BMW Car IT GmbH
 * Copyright (C) 2011 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
 * Copyright (C) 2014 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <iostream>
#include <sstream>
#include <signal.h>
#include <unistd.h>
#include "WLContext.h"
#include "WLEGLSurface.h"
#include "WLInputRenderer.h"

#define LAYER_ID 1000
#define SURFACE_ID 5200
#define WIDTH 1024
#define HEIGHT 768
#define DEF_X 0
#define DEF_Y 0
#define MAX_LAYER_ID 10000
#define MAX_SURFACE_ID 10000
#define MAX_WIDTH 1024
#define MAX_HEIGHT 768
#define MAX_X 1024
#define MAX_Y 768

using namespace std;

int gRunLoop = 0;
int gNeedRedraw = 0;
int gNeedBlank = 0;

static void SigFunc(int sig)
{
    (void)sig;
    gRunLoop = 0;
}

void usage(string& name)
{
    cerr << "Expected Arguments for " << name << endl;
    cerr << "Usage: " 
         << name << " [options]" << endl << "     "
         "Where options are:" << endl << "     "
         "-l<layer ID> (default " << LAYER_ID << ", "
         "max " << MAX_LAYER_ID << ")" << endl << "     "
         "-s<surface ID> (default " << SURFACE_ID << ", "
         "max " << MAX_SURFACE_ID << ")" << endl << "     "
         "-w<surface width> (default " << WIDTH << ", "
         "max " << MAX_WIDTH << ")" << endl << "     "
         "-h<surface height> (default " << HEIGHT << ", "
         "max " << MAX_HEIGHT << ")" << endl << "     "
         "-x<x surface position> (default " << DEF_X << ", "
         "max " << MAX_WIDTH << ")" << endl << "     "
         "-y<y surface position> (default " << DEF_Y << ", "
         "max " << MAX_HEIGHT << ")" << endl << "     "
         "-W<layer width> (default " << WIDTH << ", "
         "max " << MAX_WIDTH << ")" << endl << "     "
         "-H<layer height> (default " << HEIGHT << ", "
         "max " << MAX_HEIGHT << ")" << endl << "     "
         "-X<x layer position> (default " << DEF_X << ", "
         "max " << MAX_WIDTH << ")" << endl << "     "
         "-Y<y layer position> (default " << DEF_Y << ", "
         "max " << MAX_HEIGHT << ")" << endl;
}

struct testConfiguration
{
    unsigned int layerID;
    unsigned int surfaceID;
    unsigned int surfaceWidth;
    unsigned int surfaceHeight;
    unsigned int surfaceX;
    unsigned int surfaceY;
    unsigned int layerWidth;
    unsigned int layerHeight;
    unsigned int layerX;
    unsigned int layerY;
    unsigned int x;
    unsigned int y;
};

void setDefaultConfiguration(testConfiguration& config)
{
    config.layerID = LAYER_ID;
    config.surfaceID = SURFACE_ID;
    config.surfaceWidth = WIDTH;
    config.surfaceHeight = HEIGHT;
    config.surfaceX = DEF_X;
    config.surfaceY = DEF_Y;
    config.layerWidth = WIDTH;
    config.layerHeight = HEIGHT;
    config.layerX = DEF_X;
    config.layerY = DEF_Y;
}

bool parseCommandLine(int argc, char* argv[] , testConfiguration& config)
{
    string exeName(argv[0]);
    bool result = true;

    while ((argc > 1) && (argv[1][0] == '-'))
    {
        switch (argv[1][1])
        {
            case 'l':
                {
                    string layerID(&argv[1][2]);
                    istringstream iss(layerID);
                    unsigned short int ID;
                    iss >> ID;
                    if (!iss.fail() && (ID <= MAX_LAYER_ID))
                    {
                        config.layerID = ID;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 's':
                {
                    string surfaceID(&argv[1][2]);
                    istringstream iss(surfaceID);
                    unsigned short int ID;
                    iss >> ID;
                    if (!iss.fail() && (ID <= MAX_SURFACE_ID))
                    {
                        config.surfaceID = ID;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'w':
                {
                    string width(&argv[1][2]);
                    istringstream iss(width);
                    unsigned short int widthVal;
                    iss >> widthVal;
                    if (!iss.fail() && (widthVal <= MAX_WIDTH))
                    {
                        config.surfaceWidth = widthVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'h':
                {
                    string height(&argv[1][2]);
                    istringstream iss(height);
                    unsigned short int heightVal;
                    iss >> heightVal;
                    if (!iss.fail() && (heightVal <= MAX_HEIGHT))
                    {
                        config.surfaceHeight = heightVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'x':
                {
                    string x(&argv[1][2]);
                    istringstream iss(x);
                    unsigned short int xVal;
                    iss >> xVal;
                    if (!iss.fail() && (xVal <= MAX_X))
                    {
                        config.surfaceX = xVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'y':
                {
                    string y(&argv[1][2]);
                    istringstream iss(y);
                    unsigned short int yVal;
                    iss >> yVal;
                    if (!iss.fail() && (yVal <= MAX_Y))
                    {
                        config.surfaceY = yVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'W':
                {
                    string width(&argv[1][2]);
                    istringstream iss(width);
                    unsigned short int widthVal;
                    iss >> widthVal;
                    if (!iss.fail() && (widthVal <= MAX_WIDTH))
                    {
                        config.layerWidth = widthVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'H':
                {
                    string height(&argv[1][2]);
                    istringstream iss(height);
                    unsigned short int heightVal;
                    iss >> heightVal;
                    if (!iss.fail() && (heightVal <= MAX_HEIGHT))
                    {
                        config.layerHeight = heightVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'X':
                {
                    string x(&argv[1][2]);
                    istringstream iss(x);
                    unsigned short int xVal;
                    iss >> xVal;
                    if (!iss.fail() && (xVal <= MAX_X))
                    {
                        config.layerX = xVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            case 'Y':
                {
                    string y(&argv[1][2]);
                    istringstream iss(y);
                    unsigned short int yVal;
                    iss >> yVal;
                    if (!iss.fail() && (yVal <= MAX_Y))
                    {
                        config.layerY = yVal;
                    }
                    else
                    {
                        result = false;
                    }
                }
            break;

            default:
                result = false;
        }

        ++argv;
        --argc;
    }

    return result;

}

int main(int argc, char **argv)
{
    WLContext* wlContext;
    WLEGLSurface* eglSurface;

    testConfiguration config;
    setDefaultConfiguration(config);

    if (!parseCommandLine(argc, argv, config))
    {
        string exeName(argv[0]);
        usage(exeName);
        exit(1);
    }

    t_ilm_layer   layerId   = config.layerID;
    t_ilm_surface surfaceId = config.surfaceID;

    argc = argc; // avoid warning
    argv = argv;

    // signal handling
    signal(SIGINT,  SigFunc);
    signal(SIGTERM, SigFunc);

    ilm_init();

    wlContext = new WLContext();
    wlContext->InitWLContext(&PointerListener, NULL, &TouchListener);

    eglSurface = new WLEGLSurface(wlContext);
    eglSurface->CreateSurface(config.surfaceWidth, config.surfaceHeight);
    eglSurface->CreateIlmSurface(&config.layerID,
                                 &config.surfaceID,
                                 config.surfaceWidth,
                                 config.surfaceHeight,
                                 config.surfaceX, config.surfaceY,
                                 config.layerWidth,
                                 config.layerHeight,
                                 config.layerX, config.layerY);

    if (!InitRenderer(config.surfaceWidth, config.surfaceHeight))
    {
        fprintf(stderr, "Failed to init renderer\n");
        return -1;
    }

    DrawTouches(eglSurface);

    // wait for input event
    gRunLoop = 1;
    gNeedRedraw = 0;
    while (gRunLoop)
    {
        WaitForEvent(wlContext->GetWLDisplay());
        if (gNeedBlank && gRunLoop)
        {
            DrawBlankScreen(eglSurface);
            gNeedBlank = 0;
        }
        if (gNeedRedraw && gRunLoop)
        {
            DrawTouches(eglSurface);
            gNeedRedraw = 0;
        }
        usleep(50);
    }

    TerminateRenderer();

    ilm_layerRemoveSurface(layerId, surfaceId);
    ilm_UpdateInputEventAcceptanceOn(surfaceId,
                                     ILM_INPUT_DEVICE_TOUCH,
                                     ILM_FALSE);
    ilm_commitChanges();
    eglSurface->DestroyIlmSurface();
    ilm_commitChanges();

    ilm_destroy();
    delete eglSurface;
    delete wlContext;

    return 0;
}
